/**************************************************************************************
Copyright (c) Hilscher Gesellschaft fuer Systemautomation mbH. All Rights Reserved.
***************************************************************************************
$Id: DrvEth_Public.h 78455 2018-10-18 07:56:21Z andreasme $:

Description:
  Drv Eth Public Header
**************************************************************************************/
#ifndef DRV_ETH_PUBLIC_H_
#define DRV_ETH_PUBLIC_H_

#ifdef __cplusplus
extern "C"
{
#endif

#include "Drv_Eth_Results.h"
#include "PS_Toolbox_Public.h"

/** additional ether defines */
enum {
  /** Use this value when registering an receive handler to catch unhandled
   * ethertypes or when registering a filter handler */
  ETH_TYPE_UNSPECIFIC  = 0x0000,
  /** IP protocol ethertype */
  ETH_TYPE_IP          = 0x0800,
  /** ARP protocol ethertype */
  ETH_TYPE_ARP         = 0x0806,
  ETH_TYPE_DLR         = 0x80E1,
  /** VLAN tagged frame ethertype */
  ETH_TYPE_VLAN        = 0x8100,
  /** Profinet protocol ethertype */
  ETH_TYPE_PNO         = 0x8892,
  ETH_TYPE_EPL         = 0x88AB,
  /** Link layer discovery protocol ethertype */
  ETH_TYPE_LLDP        = 0x88CC,
  /** Media redundancy protocol ethertype */
  ETH_TYPE_MRP         = 0x88E3,
};

typedef uint8_t ETH_MAC_ADDR_T[6];

/* Ethernet frame header structure */
typedef __PACKED_PRE struct __PACKED_POST
{
  ETH_MAC_ADDR_T    tDstAddr;
  ETH_MAC_ADDR_T    tSrcAddr;
  uint16_t          usType;
}  ETH_HDR_T;

/* Ethernet frame header structure VLAN tagged */
typedef __PACKED_PRE struct __PACKED_POST
{
  ETH_MAC_ADDR_T    tDstAddr;    /**< Destination MAC address (DA)     */
  ETH_MAC_ADDR_T    tSrcAddr;    /**< Source MAC address (SA)          */
  uint16_t          usVlanTag;
  uint16_t          usVlanCtrl;
  uint16_t          usType;      /**< Frame length/type (LT)           */
}  ETH_HDR_VLAN_T;

/* Ethernet frame structure */
typedef __PACKED_PRE struct __PACKED_POST
{
  ETH_HDR_T  tHdr;
  uint8_t    abData[1504];
}  ETH_FRAME_T;

/* Ethernet frame structure with VLAN tag */
typedef __PACKED_PRE struct __PACKED_POST
{
  ETH_HDR_VLAN_T  tHdr;
  uint8_t    abData[1500];
}  ETH_FRAME_VLAN_T;

/** Forward declarations needed for some callbacks function type definitions */
struct DRV_ETH_APP_Ttag;
struct ETH_FRAME_PROP_Ttag;
struct DRV_ETH_EVENT_LISTENER_Ttag;

/** Handle to ethernet driver (opaque structure) */
typedef struct DRV_ETH_Ttag*             DRV_ETH_H;

/** Handle to ethernet load limiter (opaque structure) */
typedef void*                            DRV_ETH_LOADLIMITER_H;

/** A bitmask of Port Numbers. Each Bit corresponds to one Port*/
typedef uint8_t ETH_PORTMASK_T;

/** Ethernet Frame Priority */
typedef uint8_t ETH_PRIO_E;

enum ETH_PRIO_Etag
{
  ETH_PRIO_0   = 0,
  ETH_PRIO_1   = 1,
  ETH_PRIO_2   = 2,
  ETH_PRIO_3   = 3,
  ETH_PRIO_4   = 4,
  ETH_PRIO_5   = 5,
  ETH_PRIO_6   = 6,
  ETH_PRIO_7   = 7,
  ETH_PRIO_NWC = 8,
};

/** Frame Buffer Fragment */
typedef struct ETH_FRAME_FRAG_Ttag
{
  /** pointer to next fragment structure*/
  struct ETH_FRAME_FRAG_Ttag*   ptNext;
  /** pointer to data of this fragment */
  uint8_t*                      pbPayload;
  /** length of this fragment */
  uint16_t                      usLen;
  /** port associated with frame. only valid in first segment */
  ETH_PORTMASK_T                bPortMask;
  /** priority associated with frame. only valid in first segment */
  ETH_PRIO_E                    bPriority;
}ETH_FRAME_FRAG_T;

/* Port related MAC address parameter */
typedef struct DRV_ETH_PRM_MAC_ADDR_Ttag
{
  /* the port */
  ETH_PORTMASK_T     bPort;   /* A bit mask of Port Numbers. Each Bit corresponds to one Port:
                                 If no bit is set, the interface/chassis MAC address is meant.

                                 0x00 = Chassis MAC address
                                 0x01 = MAC address of port 0
                                 0x02 = MAC address of port 1 */
  /* the macaddress */
  uint8_t            abMac[6];
}DRV_ETH_PRM_MAC_ADDR_T;

/** Ethernet receive callback function type
 *
 * Invoked by Ethernet Driver if a frame with matching ether type
 * was received. The callback must return true if the frame will be handled
 * by application and false if the frame will not be handled by the application.
 *
 * If the callback returns false, the application must not release or touch the frame
 * in any way. On the other side, if the application returns true, the
 * application is responsible for releasing the frame.
 */
typedef bool (*DRV_ETH_RECV_CLB)(struct DRV_ETH_APP_Ttag* ptApp, ETH_FRAME_FRAG_T *ptBuffer, struct ETH_FRAME_PROP_Ttag *ptFrameProp);

/** send confirmation callback invoked if a frame was sent with enabled send
 * confirmation
 *
 * @param ptApp the application associated with the request
 * @param ptFrameProp the frame properties structure provided in send request
 * */
typedef void (*DRV_ETH_CONF_CLB)(struct DRV_ETH_APP_Ttag *ptApp, struct ETH_FRAME_PROP_Ttag *ptFrameProp);

/** Frame properties flags to be used in structure ETH_FRAME_PROP_T.bFlags */
#define ETH_FRAME_PROP_FLAG_TIMESTAMP    0x01

/** Frame Property Parameter
 *
 * This structure is used to carry additional parameters of a frame. It
 * is used for Send Confirmations and used in Receive Callbacks. */
typedef struct ETH_FRAME_PROP_Ttag
{
  /** length of this structure. Set to sizeof(ETH_FRAME_PROP_T) */
  uint8_t           bLength;
  /** flags of frame properties. See ETH_FRAME_PROP_FLAGS_* defines */
  uint8_t           bFlags;
  /** unit of the timestamp field */
  uint8_t           bTimestampUnitControlled;
  /** unit of the timestamp field */
  uint8_t           bTimestampUnitUncontrolled;
  /** Timestamp Controlled*/
  uint64_t          ullTimestampControlled;
  /** Timestamp Uncontrolled*/
  uint64_t          ullTimestampUncontrolled;
  /** send confirmation callback will be called when switch confirms send of frame */
  DRV_ETH_CONF_CLB  pfnCnfClb;
}ETH_FRAME_PROP_T;

/** Structure to hold registered applications */
typedef struct DRV_ETH_APP_Ttag
{
  /** length of this structure. Set to sizeof(DRV_ETH_APP_T) */
  uint_least8_t     bLength;

  uint_least8_t     bReserved;

  uint_least16_t    usEtherType;

  PS_SINGLY_LST_T   tListAnchor;

  DRV_ETH_H         hDrvEth;

  DRV_ETH_RECV_CLB  pfnRecvCallback;
}DRV_ETH_APP_T;

/** ethernet event enumeration (use int32_t as base data type in order to binary compatible with old versions) */
typedef int32_t ETH_EVENT_E;

enum ETH_EVENT_Etag
{
  ETH_EVENT_RESERVED     = 0,
  ETH_EVENT_LINK_CHANGED = 1,
  ETH_EVENT_MAC_CHANGED  = 2,
};

/** ethernet event arguments */
typedef struct ETH_EVENT_ARG_LINK_CHANGED_Ttag
{
  ETH_PORTMASK_T bPortMask;
}ETH_EVENT_ARG_LINK_CHANGED_T;

typedef DRV_ETH_PRM_MAC_ADDR_T ETH_EVENT_ARG_MAC_CHANGED_T;

typedef union ETH_EVENT_ARG_Ttag
{
  ETH_EVENT_ARG_LINK_CHANGED_T         tLinkChanged;

  ETH_EVENT_ARG_MAC_CHANGED_T          tChangedMac;
}ETH_EVENT_ARG_T;

/** Event callback function type
 *
 * Invoked by Ethernet Driver if a special event has occurred.
 */
typedef void (*DRV_ETH_EVENT_CLB)(struct DRV_ETH_EVENT_LISTENER_Ttag *ptListener, ETH_EVENT_E eEvent, void *ptArg, size_t ulSize);

/** Event registration structure */
typedef struct DRV_ETH_EVENT_LISTENER_Ttag
{
  uint_least8_t     bSize;

  PS_SINGLY_LST_T   tListAnchor;

  DRV_ETH_EVENT_CLB pfnClb;
}DRV_ETH_EVENT_LISTENER_T;

/** phy handling (opaque structure) */
typedef struct DRV_ETH_PHY_Ttag *DRV_ETH_PHY_H;

/**  enumeration of mau types according IANA (use int32_t as base data type in order to binary compatible with old versions) */
typedef int32_t ETH_MAU_TYPE_E;

enum ETH_MAU_TYPE_Etag
{
  /** Unkown/ */
  ETH_MAU_TYPE_UNKNOWN            = 0,
  /** 10 MBit Half Duplex Coper */
  ETH_MAU_TYPE_10BASET_HD         = 10,
  /** 10 MBit Full Duplex Coper */
  ETH_MAU_TYPE_10BASET_FD         = 11,
  /** 100 MBit Full Half Coper */
  ETH_MAU_TYPE_100BASETX_HD       = 15,
  /** 100 MBit Full Duplex Coper */
  ETH_MAU_TYPE_100BASETX_FD       = 16,
  /** 100 MBit Half Duplex Glass Fiber */
  ETH_MAU_TYPE_100BASEFX_HD       = 17,
  /** 100 MBit Full Duplex Glass Fiber */
  ETH_MAU_TYPE_100BASEFX_FD       = 18,
  /** 100 MBit Full Duplex Plastic Fiber.
   *
   *  This is a Profinet Specific Mau Type
   *  which conflicts with other iana mau types
   *  caused a lot of fun */
  ETH_MAU_TYPE_PNIO_POF_FD         = 54,
};

/** possible mdi modes (use int32_t as base data type in order to binary compatible with old versions) */
typedef int32_t ETH_MDI_MODE_E;

enum ETH_MDI_MODE_Etag
{
  ETH_MDI_MODE_RESERVED = 0,
  ETH_MDI_MODE_AUTO     = 1,
  ETH_MDI_MODE_MDI      = 2,
  ETH_MDI_MODE_MDIX     = 3,
};

/** phy properties structure */
typedef struct ETH_PHY_PROP_Ttag
{
  /** Zero terminated list of mau types supported by this phy */
  const ETH_MAU_TYPE_E    *paeSupportedMauTypes;
  /** Zero terminated list of mau types activated in phy (autonegotation mode only) */
  const ETH_MAU_TYPE_E    *paeActiveMauTypes;
  /** currentyle active mau */
  ETH_MAU_TYPE_E           eActiveMau;
  /** fiber optic mau. non zero if fiber optic physics attached */
  ETH_MAU_TYPE_E           eFiberMau;
  /** Internal use only */
  ETH_MAU_TYPE_E           aeMode[2];
  /** Non-zero if autonegotiation is in use */
  uint8_t                  bAutoneg;
}ETH_PHY_PROP_T;

/** ioctl definitions */
typedef enum DRV_ETH_IOCTL_Etag
{
  /* Generic IOCTLs */
  DRV_ETH_IOCTL_GET_INFO                        = 0x001, /**< uses DRV_ETH_IOCTL_PRM_INFO_T as parameter */
  DRV_ETH_IOCTL_SET_MACADDRESS                  = 0x002, /**< uses DRV_ETH_PRM_MAC_ADDR_T as parameter */
  DRV_ETH_IOCTL_GET_MACADDRESS                  = 0x003, /**< uses DRV_ETH_PRM_MAC_ADDR_T as parameter */
  DRV_ETH_IOCTL_GET_STATISTICS                  = 0x004, /**< uses DRV_ETH_IOCTL_PRM_STATISTICS_T as parameter */
  DRV_ETH_IOCTL_ENABLE_MCAST                    = 0x005, /**< uses DRV_ETH_PRM_MAC_ADDR_T as parameter */
  DRV_ETH_IOCTL_DISABLE_MCAST                   = 0x006, /**< uses DRV_ETH_PRM_MAC_ADDR_T as parameter */
  DRV_ETH_IOCTL_ENABLE_MCAST_FRWD               = 0x007, /**< uses DRV_ETH_IOCTL_PRM_MCAST_FRWD_T as parameter */
  DRV_ETH_IOCTL_DISABLE_MCAST_FRWD              = 0x008, /**< uses DRV_ETH_IOCTL_PRM_MCAST_FRWD_T as parameter */
  DRV_ETH_IOCTL_GET_SWITCH_DIAG                 = 0x009, /**< uses DRV_ETH_IOCTL_PRM_SWITCH_DIAG_T as parameter */
  DRV_ETH_IOCTL_SET_PORTSTATUS                  = 0x00A, /**< uses DRV_ETH_IOCTL_PRM_PORTSTATUS_T as parameter */
  DRV_ETH_IOCTL_CLEAR_FDB                       = 0x00B, /**< Does not use any parameters */
  DRV_ETH_IOCTL_GET_PORT_DELAY                  = 0x00C, /**< uses DRV_ETH_IOCTL_PRM_PORT_DELAYS_T as parameter */
  DRV_ETH_IOCTL_SET_MODE                        = 0x00D, /**< uses DRV_ETH_IOCTL_PRM_MODE_T as parameter */
  DRV_ETH_IOCTL_REGISTER_LOADCALLBACK           = 0x00E, /**< uses DRV_ETH_IOCTL_PRM_LOADCALLBACK_T as parameter */
  DRV_ETH_IOCTL_UNREGISTER_LOADCALLBACK         = 0x00F, /**< Does not use any parameters */
  /** Start operation of network switch */
  DRV_ETH_IOCTL_START                           = 0x010, /**< uses DRV_ETH_IOCTL_PRM_START_T as parameter */
  /** Stop operation of network switch. Do not
   * release resources(!) */
  DRV_ETH_IOCTL_STOP                            = 0x011, /**< Does not use any parameters */
  DRV_ETH_IOCTL_GET_ETHERLIKE_STATISTICS        = 0x012, /**< uses DRV_ETH_IOCTL_PRM_ETHERLIKE_STATISTICS_T as parameter */
  DRV_ETH_IOCTL_SET_INGRESS_EGRESS_LATENCY      = 0x013, /**< uses DRV_ETH_IOCTL_PRM_SET_INGRESS_EGRESS_LATENCY_T as parameter */

  /* Extend ioctl ranges, mostly defined by 2-port-switch */
  DRV_ETH_IOCTL_DLR_FIRST                       = 0x100,
  DRV_ETH_IOCTL_DLR_LAST                        = 0x10F,

  DRV_ETH_IOCTL_PTP_FIRST                       = 0x110,
  DRV_ETH_IOCTL_PTP_LAST                        = 0x11F,

  /* Profinet ioctl range */
  DRV_ETH_IOCTL_PN_FIRST                        = 0x200,
  DRV_ETH_IOCTL_PN_LAST                         = 0x2FF,

  /* Powerlink ioctl command range */
  DRV_ETH_IOCTL_EPL_FIRST                       = 0x300,
  DRV_ETH_IOCTL_EPL_LAST                        = 0x3FF,
}DRV_ETH_IOCTL_E;

/** Flags to be used in DRV_ETH_IOCTL_PRM_INFO_T.bFlags */
#define DRV_ETH_IOCTL_PRM_INFO_FLAG_SPLITBUFFER   0x01 /**< The ethernet driver uses split buffer mode */
#define DRV_ETH_IOCTL_PRM_INFO_FLAG_TIMESTAMP     0x02 /**< The ethernet driver support time stamping */

/** Parameters for DRV_ETH_IOCTL_GET_INFO */
typedef struct DRV_ETH_IOCTL_PRM_INFO_Ttag
{
  uint32_t  ulSpeed;

  uint16_t  usMtu;

  uint8_t   bNumPorts;

  uint8_t   bFlags; /* See DRV_ETH_IOCTL_PRM_INFO_FLAG_* defines */
}DRV_ETH_IOCTL_PRM_INFO_T;

/** Parameters for the IOCTLs
 * DRV_ETH_IOCTL_SET_MACADDRESS
 * DRV_ETH_IOCTL_GET_MACADDRESS
 * DRV_ETH_IOCTL_ENABLE_MCAST
 * DRV_ETH_IOCTL_DISABLE_MCAST */
typedef DRV_ETH_PRM_MAC_ADDR_T DRV_ETH_IOCTL_PRM_MAC_ADDR_T;

/** Flags to be used in DRV_ETH_IOCTL_PRM_STATISTICS_T.ulValidFlags */
#define DRV_ETH_IOCTL_PRM_STATISTICS_VALID_FLAG_IN_OCTETS           0x0001
#define DRV_ETH_IOCTL_PRM_STATISTICS_VALID_FLAG_IN_UCAST_PKTS       0x0002
#define DRV_ETH_IOCTL_PRM_STATISTICS_VALID_FLAG_IN_N_UCAST_PKTS     0x0004
#define DRV_ETH_IOCTL_PRM_STATISTICS_VALID_FLAG_IN_DISCARDS         0x0008
#define DRV_ETH_IOCTL_PRM_STATISTICS_VALID_FLAG_IN_ERRORS           0x0010
#define DRV_ETH_IOCTL_PRM_STATISTICS_VALID_FLAG_IN_UNKNOWN_PROTOS   0x0020
#define DRV_ETH_IOCTL_PRM_STATISTICS_VALID_FLAG_OUT_OCTETS          0x0040
#define DRV_ETH_IOCTL_PRM_STATISTICS_VALID_FLAG_OUT_UCAST_PKTS      0x0080
#define DRV_ETH_IOCTL_PRM_STATISTICS_VALID_FLAG_OUT_N_UCAST_PKTS    0x0100
#define DRV_ETH_IOCTL_PRM_STATISTICS_VALID_FLAG_OUT_DISCARDS        0x0200
#define DRV_ETH_IOCTL_PRM_STATISTICS_VALID_FLAG_OUT_ERRORS          0x0400

/** Parameters for DRV_ETH_IOCTL_GET_STATISTICS
 * The structure includes the Ethernet statistics according to RFC 1213 MIB-II Management Information Base */
typedef struct DRV_ETH_IOCTL_PRM_STATISTICS_Ttag
{
  ETH_PORTMASK_T bPortMask;   /**< Bit number of the Port that statistic data is from */

  uint32_t ulValidFlags;      /**< Flag field of valid counter values. See DRV_ETH_IOCTL_PRM_STATISTICS_VALID_FLAG_* defines */

  uint32_t ulInOctets;        /**< The total number of octets received on the
                                   interface, including framing characters. */

  uint32_t ulInUcastPkts;     /**< The number of subnetwork-unicast packets
                                   delivered to a higher-layer protocol. */

  uint32_t ulInNUcastPkts;    /**< The number of non-unicast (i.e., subnetwork-
                                   broadcast or subnetwork-multicast) packets
                                   delivered to a higher-layer protocol. */

  uint32_t ulInDiscards;      /**< The number of inbound packets which were chosen
                                   to be discarded even though no errors had been
                                   detected to prevent their being deliverable to a
                                   higher-layer protocol. */

  uint32_t ulInErrors;        /**< The number of inbound packets that contained
                                   errors preventing them from being deliverable to a
                                   higher-layer protocol. */

  uint32_t ulInUnkownProtos;  /**< The number of packets received via the interface
                                   which were discarded because of an unknown or
                                   unsupported protocol. */

  uint32_t ulOutOctets;       /**< The total number of octets transmitted out of the
                                   interface, including framing characters. */

  uint32_t ulOutUcastPkts;    /**< The total number of packets that higher-level
                                   protocols requested be transmitted to a
                                   subnetwork-unicast address, including those that
                                   were discarded or not sent. */

  uint32_t ulOutNUcastPkts;   /**< The total number of packets that higher-level
                                   protocols requested be transmitted to a non-
                                   unicast (i.e., a subnetwork-broadcast or
                                   subnetwork-multicast) address, including those
                                   that were discarded or not sent. */

  uint32_t ulOutDiscards;     /**< The number of outbound packets which were chosen
                                   to be discarded even though no errors had been
                                   detected to prevent their being transmitted. */

  uint32_t ulOutErrors;       /**< The number of outbound packets that could not be
                                   transmitted because of errors. */
}DRV_ETH_IOCTL_PRM_STATISTICS_T;

/** Flags to be used in DRV_ETH_IOCTL_PRM_ETHERLIKE_STATISTICS_T.ulValidFlags */
#define DRV_ETH_IOCTL_PRM_ETHERLIKE_STATISTICS_VALID_FLAG_ALIGNMENT_ERRORS                0x0001
#define DRV_ETH_IOCTL_PRM_ETHERLIKE_STATISTICS_VALID_FLAG_FCS_ERRORS                      0x0002
#define DRV_ETH_IOCTL_PRM_ETHERLIKE_STATISTICS_VALID_FLAG_SINGLE_COLLISION_FRAMES         0x0004
#define DRV_ETH_IOCTL_PRM_ETHERLIKE_STATISTICS_VALID_FLAG_MULTIPLE_COLLISION_FRAMES       0x0008
#define DRV_ETH_IOCTL_PRM_ETHERLIKE_STATISTICS_VALID_FLAG_SQE_TEST_ERRORS                 0x0010
#define DRV_ETH_IOCTL_PRM_ETHERLIKE_STATISTICS_VALID_FLAG_DEFERRED_TRANSMISSIONS          0x0020
#define DRV_ETH_IOCTL_PRM_ETHERLIKE_STATISTICS_VALID_FLAG_LATE_COLLISIONS                 0x0040
#define DRV_ETH_IOCTL_PRM_ETHERLIKE_STATISTICS_VALID_FLAG_ECXESSIVE_COLLISIONS            0x0080
#define DRV_ETH_IOCTL_PRM_ETHERLIKE_STATISTICS_VALID_FLAG_INTERNAL_MAC_TRANSMIT_ERRORS    0x0100
#define DRV_ETH_IOCTL_PRM_ETHERLIKE_STATISTICS_VALID_FLAG_CARRIER_SENSE_ERRORS            0x0200
#define DRV_ETH_IOCTL_PRM_ETHERLIKE_STATISTICS_VALID_FLAG_FRAME_TOO_LONGS                 0x0400
#define DRV_ETH_IOCTL_PRM_ETHERLIKE_STATISTICS_VALID_FLAG_INTERNAL_MAC_RECEIVE_ERRORS     0x0800

/** Parameters for DRV_ETH_IOCTL_GET_ETHERLIKE_STATISTICS
 * The structure includes the Ethernet-like Statistics according to RFC 1643 */
typedef struct DRV_ETH_IOCTL_PRM_ETHERLIKE_STATISTICS_Ttag
{
  ETH_PORTMASK_T bPortMask;               /**< Bit number of the Port that statistic data is from */

  uint32_t ulValidFlags;                  /**< Flag field of valid counter values. See DRV_ETH_IOCTL_PRM_ETHERLIKE_STATISTICS_VALID_FLAG_* defines */

  uint32_t ulAlignmentErrors;             /**< A count of frames received on a particular
                                               interface that are not an integral number of
                                               octets in length and do not pass the FCS check. */
  uint32_t ulFCSErrors;                   /**< A count of frames received on a particular
                                               interface that are an integral number of octets
                                               in length but do not pass the FCS check.*/
  uint32_t ulSingleCollisionFrames;       /**< A count of successfully transmitted frames on
                                               a particular interface for which transmission
                                               is inhibited by exactly one collision. */
  uint32_t ulMultipleCollisionFrames;     /**< A count of successfully transmitted frames on
                                               a particular interface for which transmission
                                               is inhibited by more than one collision. */
  uint32_t ulSQETestErrors;               /**< A count of times that the SQE TEST ERROR
                                               message is generated by the PLS sublayer for a
                                               particular interface.*/
  uint32_t ulDeferredTransmissions;       /**< A count of frames for which the first
                                               transmission attempt on a particular interface
                                               is delayed because the medium is busy.*/
  uint32_t ulLateCollisions;              /**< The number of times that a collision is
                                               detected on a particular interface later than
                                               512 bit-times into the transmission of a
                                               packet.*/
  uint32_t ulExcessiveCollisions;         /**< A count of frames for which transmission on a
                                               particular interface fails due to excessive
                                               collisions.*/
  uint32_t ulInternalMacTransmitErrors;   /**< A count of frames for which transmission on a
                                               particular interface fails due to an internal
                                               MAC sublayer transmit error.*/
  uint32_t ulCarrierSenseErrors;          /**< The number of times that the carrier sense
                                               condition was lost or never asserted when
                                               attempting to transmit a frame on a particular
                                               interface.*/
  uint32_t ulFrameTooLongs;               /**< A count of frames received on a particular
                                               interface that exceed the maximum permitted
                                               frame size.*/
  uint32_t ulInternalMacReceiveErrors;    /**< A count of frames for which reception on a
                                               particular interface fails due to an internal
                                               MAC sublayer receive error.*/
}DRV_ETH_IOCTL_PRM_ETHERLIKE_STATISTICS_T;

/** Parameters for DRV_ETH_IOCTL_ENABLE_MCAST and DRV_ETH_IOCTL_DISABLE_MCAST_FRWD */
typedef struct DRV_ETH_IOCTL_PRM_MCAST_FRWD_Ttag
{
  ETH_PORTMASK_T bPortMaskFrom;

  ETH_PORTMASK_T bPortMaskTo;

  ETH_MAC_ADDR_T tMacAddr;
}DRV_ETH_IOCTL_PRM_MCAST_FRWD_T;

/** Parameters for DRV_ETH_IOCTL_GET_SWITCH_DIAG */
typedef struct DRV_ETH_IOCTL_PRM_SWITCH_DIAG_Ttag
{
  /** Length of buffer: Application shall fill in maximum available length,
   *                    Drv_Eth will return used buffer size */
  uint16_t usLenBuffer;

  uint16_t usVersion;

  uint32_t aulBuffer[];
}DRV_ETH_IOCTL_PRM_SWITCH_DIAG_T;

/** Ethernet port states to be used in DRV_ETH_IOCTL_PRM_PORTSTATUS_T.bState */
typedef uint8_t DRV_ETH_PORTSTATE_E;
enum DRV_ETH_PORTSTATE_Etag
{
  DRV_ETH_PORTSTATE_DISCARDING = 0,   /**< All frames are discarded */
  DRV_ETH_PORTSTATE_BLOCKED    = 1,   /**< Network Control frames are received and forwarded, all other frames are discarded */
  DRV_ETH_PORTSTATE_FORWARDING = 2    /**< All frames are received and forwarded */
};

/** Parameters for DRV_ETH_IOCTL_SET_PORTSTATUS */
typedef struct DRV_ETH_IOCTL_PRM_PORTSTATUS_Ttag
{
  /* the port */
  ETH_PORTMASK_T     bPort;
  /* port state. See DRV_ETH_PORTSTATE_* */
  uint8_t            bState;
}DRV_ETH_IOCTL_PRM_PORTSTATUS_T;

/** Parameters for DRV_ETH_IOCTL_GET_PORT_DELAY */
typedef struct DRV_ETH_IOCTL_PRM_PORT_DELAYS_Ttag
{
  /* the port */
  ETH_PORTMASK_T     bPort;
  /* the timestamp receive offset in ns */
  uint16_t           usRxOffset1ns;
  /* the timestamp transmit offset in ns */
  uint16_t           usTxOffset1ns;
}DRV_ETH_IOCTL_PRM_PORT_DELAYS_T;


/** Driver ethernet mode */
#define DRV_ETH_IOCTL_PRM_MODE_FLAG_NONSTOP_MODE  0x01   /**< Non-Stop Mode Active
                                                           * Drv Eth will release the frame if the application
                                                           * was not able to handle it. Thus Drv_Eth will never
                                                           * wait for any application. This is mainly sensible
                                                           * for online debugging purposes
                                                           */

/** Parameters for DRV_ETH_IOCTL_SET_MODE */
typedef struct DRV_ETH_IOCTL_PRM_MODE_Ttag
{
  /* the mode flags to apply. See DRV_ETH_IOCTL_PRM_MODE_FLAG_* defines */
  uint8_t     bFlags;
}DRV_ETH_IOCTL_PRM_MODE_T;

/** Load callback function type */
typedef void (*DRV_ETH_LOAD_CLB)(void *pvArgument);

/** Parameters for DRV_ETH_IOCTL_REGISTER_LOADCALLBACK
 * This is used for system load protection */
typedef struct DRV_ETH_IOCTL_PRM_LOADCALLBACK_Ttag
{
  DRV_ETH_LOAD_CLB pfnLoadClb;

  void            *pvArgument;
}DRV_ETH_IOCTL_PRM_LOADCALLBACK_T;

/** Parameters for DRV_ETH_IOCTL_START */
typedef struct DRV_ETH_IOCTL_PRM_START_Ttag
{
  PS_WORKER_H     hContext;
}DRV_ETH_IOCTL_PRM_START_T;

/** Parameters for DRV_ETH_IOCTL_SET_INGRESS_EGRESS_LATENCY */
typedef struct DRV_ETH_IOCTL_PRM_SET_INGRESS_EGRESS_LATENCY_Ttag
{
  uint32_t uiIngressLatency100MBits;
  uint32_t uiIngressLatency10MBits;
  uint32_t uiEgressLatency100MBits;
  uint32_t uiEgressLatency10MBits;
}DRV_ETH_IOCTL_PRM_SET_INGRESS_EGRESS_LATENCY_T;

/** Union of all generic IOCTL parameters. To be used as parameter pvPrm when calling the function DrvEth_Ioctl */
typedef union DRV_ETH_IOCTL_PRM_Ttag
{
  DRV_ETH_IOCTL_PRM_INFO_T                       tInfo;
  DRV_ETH_IOCTL_PRM_MAC_ADDR_T                   tSetMac;
  DRV_ETH_IOCTL_PRM_MAC_ADDR_T                   tGetMac;
  DRV_ETH_IOCTL_PRM_STATISTICS_T                 tStat;
  DRV_ETH_IOCTL_PRM_ETHERLIKE_STATISTICS_T       tEtherLikeStat;
  DRV_ETH_IOCTL_PRM_MCAST_FRWD_T                 tEnableMCastFrwd;
  DRV_ETH_IOCTL_PRM_MCAST_FRWD_T                 tDisableMCastFrwd;
  DRV_ETH_IOCTL_PRM_SWITCH_DIAG_T                tSwitchDiag;
  DRV_ETH_IOCTL_PRM_PORTSTATUS_T                 tPortState;
  DRV_ETH_IOCTL_PRM_MAC_ADDR_T                   tEnableMCast;
  DRV_ETH_IOCTL_PRM_MAC_ADDR_T                   tDisableMCast;
  DRV_ETH_IOCTL_PRM_PORT_DELAYS_T                tPortDelay;
  DRV_ETH_IOCTL_PRM_MODE_T                       tSetMode;
  DRV_ETH_IOCTL_PRM_LOADCALLBACK_T               tRegisterLoadClb;
  DRV_ETH_IOCTL_PRM_START_T                      tStart;
  DRV_ETH_IOCTL_PRM_SET_INGRESS_EGRESS_LATENCY_T tIngressEgressDelays;
}DRV_ETH_IOCTL_PRM_T;

uint32_t          DrvEth_RegisterEtherType  (DRV_ETH_APP_T *ptApp);
void              DrvEth_UnregisterEtherType(DRV_ETH_APP_T *ptApp);

uint32_t          DrvEth_RegisterFilter     (DRV_ETH_APP_T *ptApp);
void              DrvEth_UnregisterFilter   (DRV_ETH_APP_T *ptApp);

/** Get free buffer from the low level.
 * NOTE: In order to be conform with the DrvEth concept, an application should only send
 * frames with the same ethertype as it has registered for.
 *
 * @param ptApp is the handle of the structure used to register the application in DrvEth_RegisterEtherType
 * @param size is the minimum size the buffer needs to have */
ETH_FRAME_FRAG_T* DrvEth_GetBuffer (DRV_ETH_APP_T *ptApp, size_t size);

/** Send frame buffer.
 * Use the function DrvEth_GetBuffer to get a free buffer before calling this function
 * NOTE: In order to be conform with the DrvEth concept, an application should only send
 * frames with the same ethertype as it has registered for.
 *
 * @param ptApp is the handle of the structure used to register the application in DrvEth_RegisterEtherType
 * @param ptBuffer is the frame buffer to be send
 * @param ptCnfProp is the confirmation parameters (optional). May be used by the application to receive a
 *        send confirmation */
void                  DrvEth_SendBuffer(DRV_ETH_APP_T *ptApp, ETH_FRAME_FRAG_T* ptBuffer, ETH_FRAME_PROP_T* ptCnfProp);

void                  DrvEth_ReleaseBuffer(DRV_ETH_APP_T *ptApp, ETH_FRAME_FRAG_T* ptBuffer);

void                  DrvEth_SubscribeEvents(DRV_ETH_H hDrv,   DRV_ETH_EVENT_LISTENER_T *ptEvent);
void                  DrvEth_UnsubscribeEvents(DRV_ETH_H hDrv, DRV_ETH_EVENT_LISTENER_T *ptEvent);


uint32_t              DrvEth_Ioctl(DRV_ETH_H hDrv, uint_fast16_t usCommand, void* pvPrm, size_t ulSize);

DRV_ETH_PHY_H         DrvEth_GetPhy(DRV_ETH_H hDrv, ETH_PORTMASK_T bPort);
ETH_MAU_TYPE_E        DrvEth_HasLink(DRV_ETH_H hDrv);

bool                  DrvEth_HasValidMacAddresses(DRV_ETH_H hDrvEth, unsigned int uiNumPorts);

void                  DrvEth_OS_AssignMacAddresses(DRV_ETH_H hDrvEth, unsigned int uiNumPorts);

DRV_ETH_LOADLIMITER_H DrvEth_OS_LoadLimiter_MonitorIdle(DRV_ETH_H hDrvEth, OSAL_TICKS_MS_T ulInterval1ms);
DRV_ETH_LOADLIMITER_H DrvEth_OS_LoadLimiter_MonitorWorker(DRV_ETH_H hDrvEth, PS_WORKER_H hWorker, OSAL_TICKS_MS_T ulInterval1ms);


ETH_MAU_TYPE_E        DrvPhy_GetCurrentMau(DRV_ETH_PHY_H hPhy);

/** Activate autonegotiation of phy.
 *
 * @param hPhy is the phy handle
 * @param pabMauTypes is an zero terminated array specifying the mau types to allow for autoneg.
 *        Set to null if autoneg shall do all supported modes of phy */
void                  DrvPhy_SetAutonegMode(DRV_ETH_PHY_H hPhy, const ETH_MAU_TYPE_E *pabMauTypes);
uint32_t              DrvPhy_SetFixedMode  (DRV_ETH_PHY_H hPhy, ETH_MAU_TYPE_E bMauType, ETH_MDI_MODE_E eMdiMode);
void                  DrvPhy_SetDisabled   (DRV_ETH_PHY_H hPhy);

const ETH_PHY_PROP_T* DrvPhy_GetProperties  (DRV_ETH_PHY_H hPhy);

/** Initialize NEC CB12 Phy driver
 *
 * This is the internal phy of netX10/netX50/netX51/netX52 and netX4000 */
uint32_t              DrvPhy_CB12_Init(DRV_ETH_H hDrvEth, DRV_ETH_PHY_H hPhy);

/** Initialize NEC CB12 Phy driver in Auto MDI fixed Mode
 *
 * This is the internal phy of netX100/netX500 */
uint32_t              DrvPhy_CB12_InitFixAutoMDI(DRV_ETH_H hDrvEth, DRV_ETH_PHY_H hPhy);

/** Initialize BroadCom 5241 phy driver
 *
 * This is the external phy used with netX5/netX6 and netX90 */
uint32_t              DrvPhy_BCM5241_Init(DRV_ETH_H hDrvEth, DRV_ETH_PHY_H hPhy);

/** Initialize OmniPhy phy driver
 *
 * This is the internal phy of netX90 MPW */
uint32_t              DrvPhy_Netx90MPW_Init(DRV_ETH_H hDrvEth, DRV_ETH_PHY_H hPhy);

/** Initialize OmniPhy phy driver
 *
 * This is the internal phy of netX90 */
uint32_t              DrvPhy_Netx90_Init(DRV_ETH_H hDrvEth, DRV_ETH_PHY_H hPhy);

/** Export drv eth as shared object.
 *
 * For use in real time ethernet NXOs to make drv eth functions
 * available for other loaded NXOs
 */
OSAL_RESULT           DrvEth_OS_ExportSharedObject(DRV_ETH_H hEthNDIS);


/** Import drv eth shared object.
 *
 * For use in NXOs which like to use drv eth functions from a real time
 * ethernet module. Link with "drv_eth_so" instead of "drv_eth"
 */
OSAL_RESULT           DrvEth_OS_ImportSharedObject(DRV_ETH_H *phEthNDIS);

#ifdef __cplusplus
}
#endif

#endif /* DRV_ETH_PUBLIC_H_ */
